DELIMITER $$
DROP PROCEDURE IF EXISTS getSunsuccessFeesData $$
CREATE PROCEDURE getSunsuccessFeesData(IN studentIds TEXT)
BEGIN

SELECT
  u.id AS studentId,
  -- Total Debit = bills + payments + cancelled credits + cancelled receipts
  (COALESCE(b.total_bills, 0)
   + COALESCE(p.total_payments, 0)
   + COALESCE(cc.c_total_credits, 0)
   + COALESCE(cr.c_total_receipts, 0)) AS TotalPayableAmount,
  -- Total Credit = credits + receipts + cancelled bills + cancelled payments
  (COALESCE(c.total_credits, 0)
   + COALESCE(r.total_receipts, 0)
   + COALESCE(cb.c_total_bills, 0)
   + COALESCE(cp.c_total_payments, 0)) AS TotalPaidAmount,
  -- Outstanding = Total Debit - Total Credit
  (
    (COALESCE(b.total_bills, 0)
     + COALESCE(p.total_payments, 0)
     + COALESCE(cc.c_total_credits, 0)
     + COALESCE(cr.c_total_receipts, 0))
    -
    (COALESCE(c.total_credits, 0)
     + COALESCE(r.total_receipts, 0)
     + COALESCE(cb.c_total_bills, 0)
     + COALESCE(cp.c_total_payments, 0))
  ) AS TotalOutstanding,
  -- Outstanding Percent = (Outstanding * 100) / Total Debit
  CASE 
    WHEN (COALESCE(b.total_bills, 0)
          + COALESCE(p.total_payments, 0)
          + COALESCE(cc.c_total_credits, 0)
          + COALESCE(cr.c_total_receipts, 0)) = 0 THEN 0
    ELSE ROUND(
      (
        (
          (COALESCE(b.total_bills, 0)
           + COALESCE(p.total_payments, 0)
           + COALESCE(cc.c_total_credits, 0)
           + COALESCE(cr.c_total_receipts, 0))
          -
          (COALESCE(c.total_credits, 0)
           + COALESCE(r.total_receipts, 0)
           + COALESCE(cb.c_total_bills, 0)
           + COALESCE(cp.c_total_payments, 0))
        ) * 100.0
      ) / NULLIF(
          (COALESCE(b.total_bills, 0)
           + COALESCE(p.total_payments, 0)
           + COALESCE(cc.c_total_credits, 0)
           + COALESCE(cr.c_total_receipts, 0)), 0
        ), 2)
  END AS OutstandingPercent
FROM users u
-- BILL RECEIVABLE (all)
LEFT JOIN (
  SELECT bh.studentId, SUM(bfps.feeAmount) AS total_bills
  FROM bill_header bh
  JOIN bill_fee_plan_rule_stage bfps ON bh.id = bfps.billId
  WHERE bh.raiseBillCategory = 'BILL_RECEIVABLE'
  GROUP BY bh.studentId
) b ON u.id = b.studentId
-- BILL PAYABLE (all)
LEFT JOIN (
  SELECT bh.studentId, SUM(bfps.feeAmount) AS total_credits
  FROM bill_header bh
  JOIN bill_fee_plan_rule_stage bfps ON bh.id = bfps.billId
  WHERE bh.raiseBillCategory = 'BILL_PAYABLE'
  GROUP BY bh.studentId
) c ON u.id = c.studentId
-- PAYMENTS (all)
LEFT JOIN (
  SELECT ph.studentId, SUM(ph.amount) AS total_payments
  FROM payment_header ph
  GROUP BY ph.studentId
) p ON u.id = p.studentId
-- RECEIPTS (all)
LEFT JOIN (
  SELECT rh.studentId, SUM(rh.amount) AS total_receipts
  FROM receipt_header rh
  GROUP BY rh.studentId
) r ON u.id = r.studentId
-- BILL RECEIVABLE (cancelled)
LEFT JOIN (
  SELECT bh.studentId, SUM(bfps.feeAmount) AS c_total_bills
  FROM bill_header bh
  JOIN bill_fee_plan_rule_stage bfps ON bh.id = bfps.billId
  WHERE bh.raiseBillCategory = 'BILL_RECEIVABLE' AND bh.billStatus = 'CANCELLED'
  GROUP BY bh.studentId
) cb ON u.id = cb.studentId
-- BILL PAYABLE (cancelled)
LEFT JOIN (
  SELECT bh.studentId, SUM(bfps.feeAmount) AS c_total_credits
  FROM bill_header bh
  JOIN bill_fee_plan_rule_stage bfps ON bh.id = bfps.billId
  WHERE bh.raiseBillCategory = 'BILL_PAYABLE' AND bh.billStatus = 'CANCELLED'
  GROUP BY bh.studentId
) cc ON u.id = cc.studentId
-- PAYMENTS (cancelled)
LEFT JOIN (
  SELECT ph.studentId, SUM(ph.amount) AS c_total_payments
  FROM payment_header ph
  WHERE ph.status = 'CANCELLED'
  GROUP BY ph.studentId
) cp ON u.id = cp.studentId
-- RECEIPTS (cancelled)
LEFT JOIN (
  SELECT rh.studentId, SUM(rh.amount) AS c_total_receipts
  FROM receipt_header rh
  WHERE rh.status = 'CANCELLED'
  GROUP BY rh.studentId
) cr ON u.id = cr.studentId
WHERE FIND_IN_SET(CAST(u.id AS CHAR), studentIds);

END $$
DELIMITER ;