/*
 * Following code defines a multi sort toolbar which can be plugged into any grid
 * using xtype
 */
Ext.define('Academia.view.criteriabuilder.grid.MultiSortToolbar', {
    extend: 'Ext.toolbar.Toolbar',
    requires: ['Ext.data.*', 'Ext.grid.*', 'Ext.util.*', 'Ext.toolbar.*', 'Ext.ux.ToolbarDroppable', 'Ext.ux.BoxReorderer'],
    alias: 'widget.multiSortToolbar',
    initComponent: function() {
        var me = this;
        me.on({
            // wait for the first layout to access the headerCt (we only want this once):
            single: true,
            autoDestroy: true,
            // tell the toolbar's droppable plugin that it accepts items from the columns' dragdrop group
            afterlayout: function(tbar) {
                var headerCt = tbar.up("grid").child("headercontainer");
                droppable.addDDGroup(headerCt.reorderer.dragZone.ddGroup);
                me.doSort();
            }
        });

        var reorderer = Ext.create('Ext.ux.BoxReorderer', {
            listeners: {
                Drop: function(r, c, button) { //update sort direction when button is dropped
                    me.changeSortDirection(button, false);
                }
            }
        });
        var droppable = Ext.create('Ext.ux.ToolbarDroppable', {
            /**
             * Creates the new toolbar item from the drop event
             */
            createItem: function(data) {
                var header = data.header,
                    headerCt = header.ownerCt,
                    reorderer = headerCt.reorderer;
                // Hide the drop indicators of the standard HeaderDropZone
                // in case user had a pending valid drop in
                if (reorderer) {
                    reorderer.dropZone.invalidateDrop();
                }
                return me.createSorterButtonConfig({
                    text: header.text,
                    arrowTooltip: 'Remove',
                    cls: 'dropeedButton',
                    iconAlign: 'left',
                    sortData: {
                        //property: header.dataIndex,
                        //direction: "ASC",
                        sortOrder: "ASC",
                        searchResultViewColumnId: header.data.id
                    }
                });
            },
            /**
             * Custom canDrop implementation which returns true if a column can be added to the toolbar
             * @param {Object} data Arbitrary data from the drag source. For a HeaderContainer, it will
             * contain a header property which is the Header being dragged.
             * @return {Boolean} True if the drop is allowed
             */
            canDrop: function(dragSource, event, data) {
                var sorters = me.getSorters(),
                    header = data.header,
                    length = sorters.length,
                    entryIndex = this.calculateEntryIndex(event),
                    targetItem = this.toolbar.getComponent(entryIndex),
                    i;
                // Group columns have no dataIndex and therefore cannot be sorted
                // If target isn't reorderable it could not be replaced
                if (!header.dataIndex || (targetItem && targetItem.reorderable === false)) {
                    return false;
                }
                for (i = 0; i < length; i++) {
                    if (sorters[i].property == header.dataIndex) {
                        return false;
                    }
                    if (sorters[i].searchResultViewColumnId == header.data.id) {
                        return false;
                    }
                }
                return true;
            },
            afterLayout: function() {
                me.doSort();
            }
        });

        Ext.applyIf(me, {
            plugins: [droppable, reorderer],
            items: [{
                xtype: 'tbtext',
                text: 'Sorting order:',
                reorderable: false
            }]
        });

        me.callParent(arguments);
    },
    /**
     * Returns an array of sortData from the sorter buttons
     * @return {Array} Ordered sort data from each of the sorter buttons
     */
    getSorters: function() {
        var sorters = [];
        Ext.each(this.query('button'), function(button) {
            sorters.push(button.sortData);
        }, this);

        return sorters;
    },
    /**
     * Convenience function for creating Toolbar Buttons that are tied to sorters
     * @param {Object} config Optional config object
     * @return {Object} The new Button configuration
     */
    createSorterButtonConfig: function(config) {
        var me = this;
        config = config || {};
        Ext.applyIf(config, {
            listeners: {
                click: function(button, e) {
                    me.changeSortDirection(button, true);
                },
                arrowclick: function(button) {
                    button.up('toolbar').remove(button);
                }
            },
            //iconCls: 'sort-direction-' + config.sortData.direction.toLowerCase(),
            iconCls: 'sort-direction-' + config.sortData.sortOrder.toLowerCase(),
            reorderable: true,
            xtype: 'splitbutton'
        });
        return config;
    },
    /**
     * Callback handler used when a sorter button is clicked or reordered
     * @param {Ext.Button} button The button that was clicked
     * @param {Boolean} changeDirection True to change direction (default). Set to false for reorder
     * operations as we wish to preserve ordering there
     */
    changeSortDirection: function(button, changeDirection) {
        var sortData = button.sortData,
            iconCls = button.iconCls;
        if (sortData) {
            if (changeDirection !== false) {
                //button.sortData.direction = Ext.String.toggle(button.sortData.direction, "ASC", "DESC");
                button.sortData.sortOrder = Ext.String.toggle(button.sortData.sortOrder, "ASC", "DESC");
                button.setIconCls(Ext.String.toggle(iconCls, "sort-direction-asc", "sort-direction-desc"));
            }
            this.up("grid").getStore().clearFilter();
            this.doSort();
        }
    },
    doSort: function() {
        this.up("grid").getStore().sort(this.getSorters());
    }

});